; --------------------------------------------------------------
; RETRO, Release 8
; --------------------------------------------------------------
;
; This is the "core" of RETRO. It defines the interpreter,
; compiler, and initial words and variables used to build the
; rest of RETRO.
;
; Documentation is presently built by examining this file. That
; will change in the near future. In the meantime, leave lines
; that begin with ';|' alone. Comment style will be changing
; over the next week or two.
; --------------------------------------------------------------



; --------------------------------------------------------------
; 'init'                                                  .forth
;
; This word should be called upon startup to compile the initial
; code that defines the RETRO language.
;
; If any errors happen during bootstrap, it calls the boot_error
; function to alert the user.
; --------------------------------------------------------------

code 'init', init_retroforth
	call reset
	upsh bootstrap
	upsh [bootstrap.size]
	call eval
	cmp dword [error], -1
	jz boot_error
next

boot_error:
	upsh boot1
	upsh [boot1.size]
	call type
next



; --------------------------------------------------------------
; 'boot'                                                  .forth
;
; This word is called upon a successful bootstrap. By default it
; displays an error message, so your code should replace it with
; something during the initial bootstrap.
; --------------------------------------------------------------

code 'boot', boot
        upsh boot0
        upsh [boot0.size]
        call type
        jmp $


; --------------------------------------------------------------
; 'key'                                                   .forth
;
; A vectored word normally used to read input from somewhere.
; --------------------------------------------------------------

code 'key', key
	jmp sys_key


; --------------------------------------------------------------
; 'emit'                                                  .forth
;
; A vectored word that normally displays a character.
; --------------------------------------------------------------

code 'emit', emit
	jmp sys_emit


; --------------------------------------------------------------
; 'type'                                                  .forth
;
; A vectored word that normally displays a string in addr/count
; format.
; --------------------------------------------------------------

code 'type', type
	jmp sys_type


; --------------------------------------------------------------
; 'forth'                                                 .forth
; 'macro'
; 'self'
;
; These words are used to switch the active class easily.
; --------------------------------------------------------------

code 'forth', dict_forth
	mov dword [class], do_forth
next
code 'macro', dict_macro
	mov dword [class], do_macro
next
code 'self', dict_self
	mov dword [class], do_self
next


; --------------------------------------------------------------
; 'find'                                                  .forth
;
; Search the dictionary for a string, returns TRUE if found,
; FALSE if not (?if checks this). Will leave either the xt or
; string on the stack depending on success/failure.
; --------------------------------------------------------------

code 'find', find	        ;
	push ebx                ;
	mov ebx, flast          ;
.o:	upop ecx                ;
	push esi		;
	mov esi,eax             ;
	mov dh,  byte [eax]     ; get first char to do quick compare:
	mov dl, cl              ; and the length; do both compares at once!
.a:	mov ebx,[ebx]           ;
	test ebx,ebx            ;
	jz .end                 ; end of wordlist
	lea edi, [ebx+13]       ; point edi at the string
	cmp dx, [edi-1]		;
	jne .a			;
.len:	push esi                ; same length, so do the compare
	push ecx                ;
	repe cmpsb              ;
	pop ecx                 ;
	pop esi                 ;
	jne .a                  ;
	mov dword [which], ebx	;
        mov eax,[ebx+8]         ; class
        mov dword [action], eax ;
	mov eax,[ebx+4]         ; exact match: return XT
	clc                     ;
.ret:   pop esi			;
	pop ebx                 ;
	ret			;
.end:	pop esi			;
	upsh ecx                ; no matches
	stc                     ;
	pop ebx			;
next


; --------------------------------------------------------------
; '>number'                                               .forth
;
; Attempt to convert a string into a number using the current
; base. Will return TRUE on success or FALSE on failure. Like
; 'find', it returns a string or the converted number.
; --------------------------------------------------------------

code '>number', number	        ;
	push dword [base]       ;
	mov ecx,eax	        ; n   (keep on stack in case of failure)
	mov ebx,[esi]	        ; a
	dup		        ;
	xor eax,eax	        ; the number
	xor edx,edx	        ; temp
	mov dl,[ebx]	        ;
	cmp dl,45	        ; -     Sign prefix
	pushf		        ;
	jne .a		        ;
	inc ebx 	        ;
	dec ecx 	        ;
	mov dl,[ebx]	        ;
.a:	sub dl,35	        ; $%&'  Base prefix
	cmp dl,5	        ;
	ja .b		        ;
	mov dl,[edx+bases]      ;
	mov byte [base],dl      ;
	inc ebx 	        ;
	dec ecx 	        ;
.b:	mov dl,[ebx]	        ; digits
	inc ebx 	        ;
	call digit	        ;
	jc .err 	        ;
	loop .b 	        ;
	jmp .c		        ;
.err:	add esp, 4	        ;
	drop		        ;
	stc		        ;
	jmp .ret	        ;
.c:	popf		        ;
	jne .d		        ;
	neg eax 	        ;
.d:	add esi,8	        ;
	clc		        ;
.ret:	pop dword [base]        ;
next

digit:	cmp byte [base],255     ;
	je .x10 	        ;
	cmp dl,57	        ; 9
	jbe .a		        ;
	and dl,5Fh	        ; uppercase
	cmp dl,65	        ; throw out chars between '9' and 'A'
	jb .err 	        ;
	sub dl,7	        ;
.a:	sub dl,48	        ; 0
	cmp dl, byte [base]     ;
	jb .x10 	        ;
.err:	stc		        ; not a digit
	ret		        ;
.x10:	imul eax,[base]         ;
	add eax,edx	        ;
	clc		        ;
next


interpret:
.o:	call query	        ; Get a WORD
.word:	upsh 32 	        ; Push a <SPACE> onto the stack
	call parse	        ; Parse until we find <SPACE>
	jnz .find	        ; Look for the end of the line
	drop		        ; DROP two numbers
	drop		        ;
	next
.find:	call find	        ; See if we can find the word
	jnc .exec	        ; Yes? Interpret it
	call number	        ; No? Then make it a number
	jnc .lit	        ; Loop back
	call notfound		; 
	jnc .word		;
	next			;
.exec:	mov edx, [action]	; Call the handler for this
	call edx		; class of words
	jmp .word	        ; And Loop back
.lit:	cmp dword [state], -1
	jnz .word
	call literal
	jmp .word


query:			        ;
	mov dword ecx,[source]  ;
	or ecx,ecx	        ;
	jnz query_mem	        ;
next
query_mem:		        ;
	mov edi,[tin]	        ;
	upsh edi	        ; Input pointer
	dup		        ;
	cmp byte [edi], 9	; TAB
	jne .notab		;
	mov byte [edi], 32	; Make the TAB a SPACE
.notab:	cmp byte [edi],10       ; Skip LF
	jne .a			;
	inc edi 	        ; Increase our pointer
.a:	mov [tin],edi	        ;
	sub ecx,edi	        ; Remaining length
	jbe .eof	        ;
	mov al,10	        ; Line Feed
	repne scasb	        ;
	mov eax,edi	        ;
	jne .b		        ;
	dec eax 	        ;
	cmp Byte [eax-1],13     ; We make CR optional
	jne .b		        ;
	dec eax 	        ;
.b:	upop [tp]	        ;
	drop		        ;
	next		        ;
.eof:	drop		        ;
	drop		        ;
	add esp,4	        ; Discard caller
	pop Dword [tp]	        ;
	pop Dword [tin]         ;
	pop Dword [source]      ;
next


; --------------------------------------------------------------
; 'eval'                                                  .forth
;
; Evaluate a string. This makes use of 'interpret' and requires
; a valid string to be passed on the stack.
; --------------------------------------------------------------

code 'eval', eval
	push Dword [source]     ; Save "source"
	push Dword [tin]        ; Save ">in"
	push Dword [tp]         ; Save "tp"
	add eax,[esi]	        ;
	upop [source]	        ; New "source"
	upop [tin]	        ; New ">in"
.a:	call query
	call interpret.word
	jmp .a


; --------------------------------------------------------------
; '1,'                                                    .forth
; '2,'
; '3,'
; ' ,'
;
; These words are used to inline between one and four bytes to
; the top of the heap.
; --------------------------------------------------------------

code ',', comma 	        ; comma (,) saves a value to "here"
	mov ecx,4	        ; By default, it uses a dword (4 bytes)
.a:	mov edx,[h]	        ;
	mov [edx],eax	        ;
	drop		        ;
	add edx,ecx	        ;
	mov [h],edx	        ;
	mov dword [tail], -1	; Disable tail recursion
next
code '1,', comma1	        ; comma1 (1,) saves 1 byte to "here"
	mov ecx,1	        ;
	jmp comma.a
code '2,', comma2	        ; comma2 (2,) saves 2 bytes to "here"
	mov ecx,2	        ;
	jmp comma.a
code '3,', comma3	        ; comma3 (3,) saves 3 bytes to "here"
	mov ecx,3	        ;
	jmp comma.a


dolit:	dup		        ; This is where we handle literals.
	mov eax,[esp]	        ;
	mov eax,[eax]	        ;
	add dword [esp],4       ;
next


; --------------------------------------------------------------
; 'entry'
;
; Create a new dictionary entry. This takes a string from the
; stack and generates an entry pointing to the top of the heap.
; The new entry is given the current class.
; --------------------------------------------------------------

code 'entry', _entry
	push ecx	        ;
	mov edi,[d]	        ; LFA
	mov ecx,[last]	        ; last
	mov edx,[ecx]	        ;
	mov [edi],edx	        ; LFA= [last]
	mov [ecx],edi	        ; last= LFA
	mov ecx,[h]	        ;
	mov [edi+4],ecx         ; CFA= here
	mov edx, [class]	;
	mov [edi+8],edx		;
	mov [edi+12],al	        ; Length
	add edi,13	        ;
	upop ecx	        ;
	push esi	        ;
	mov esi,eax	        ;
	rep movsb	        ;
	mov [d],edi	        ; d= d+9+length
	pop esi 	        ;
	pop ecx 	        ;
	drop
next


; --------------------------------------------------------------
; ']'                                                     .forth
;
; Set state to -1 (compilation mode)
; --------------------------------------------------------------

code ']', rbracket
	mov dword [state], -1
next


; --------------------------------------------------------------
; Undo the last compilation
; Used by:  ]
; Notes:    Resets HERE, and LAST to their prior state
; --------------------------------------------------------------

undo_compile:			;
	mov ebx, dword [old_h]	; Reset the heap pointer
	mov dword [h], ebx	;
	mov ebx, dword [last]	; Reset the dictionary pointer
	mov ebx, [ebx]		; (Same as "last @ @ last !")
	mov dword [last], ebx	;
	add esp, 4		; 
	mov ebx, $c3c3c3c3	; We do a patch to make sure
	mov ecx, dword [h]	; that attempts to call the
	mov [ecx], ebx		; bad definition fail.
next


; --------------------------------------------------------------
; 'compile'                                               .forth
;
; Compile a relative call to an address passed on the stack
; --------------------------------------------------------------

code 'compile', compile         ; This routine compiles in CALL's to
	sub eax,[h]	        ; words. Address is passed in EAX
	sub eax,5	        ;
	upsh 0xE8	        ; CALL opcode
	call comma1	        ; 0xE8
	call comma	        ; ADDRESS
	mov dword [tail], 0	; Enable tail-call
next


; --------------------------------------------------------------
; '['                                                     .macro
;
; Set state to 0, (interpreter mode)
; --------------------------------------------------------------

mcode '[', lbracket	        ; Switch back to the interpreter
	mov dword [state], 0
next


; --------------------------------------------------------------
; ';;'                                                    .macro
;
; Compile an exit (ret instruction) into the current word. If
; the last thing compiled was a call, this will make it a jump
; rather than compiling in the exit instruction. 
; --------------------------------------------------------------

mcode ';;', ssemi	        ; Exit a word (; will call this!)
	cmp dword [tail], 1	;
	jz .b			;
	mov edx,[h]	        ;
	sub edx,5	        ;
	cmp byte [edx],0xE8     ; Was the last thing compiled a CALL?
	jnz .a		        ; No, skip the next line
	cmp dword [tail],0	; See if we should compile a tail-call
	jnz .a			; No? skip it and compile a ret
	inc byte [edx]	        ; Yes, change to JMP
	mov dword [tail], 1	;
	next		        ; and exit
.a:	mov byte [edx+5],0xC3   ; If not a CALL, compile in a RET
	inc dword [h]	        ;
	mov dword [tail], -1	;
.b:	next


; --------------------------------------------------------------
; ';'                                                     .macro
;
; End the current definition. Calls ';;' and '['
; --------------------------------------------------------------

mcode ';', semi ; Compile in an exit to the current word
	call ssemi	        ; And go back to the interpreter
	jmp lbracket	        ;


; --------------------------------------------------------------
; ':'                                                      .self
;
; Create a new named defintion. Will start the compiler (see 
; ']' for what's involved in this).
; --------------------------------------------------------------

self ':', colon 	        ; Ok, this is the entry to the compiler
	mov ebx, dword [h]	;
	mov dword [old_h], ebx	;
	mov dword [tail], -1	;
	upsh 32			;
	call parse		;
	call _entry	        ; * Create a new word
	jmp rbracket	        ; * And jump to the real compiler


; --------------------------------------------------------------
; 'literal'                                               .macro
;
; Compile a literal (number) into a definition.
; --------------------------------------------------------------

mcode 'literal', literal        ; Compile in a literal
	upsh dolit	        ;
	call compile	        ;
	call comma	        ;
next


; --------------------------------------------------------------
; '#'                                                     .forth
;
; Display an unsigned number with no trailing space
; --------------------------------------------------------------

code '#', print 	        ;
	push edx	        ;
	push edi	        ;
	mov edi,esi	        ; edi = buffer (in stack space)
	sub edi,4	        ;
.a:	xor edx,edx	        ;
	div dword [base]        ;
	add dl,'0'	        ;
	cmp dl,'9'	        ;
	jbe .b		        ;
	add dl,7+32	        ;
.b:	dec edi 	        ;
	mov [edi],dl	        ;
	or eax,eax	        ;
	jnz .a		        ;
	mov eax,edi	        ; Print
	upsh esi	        ;
	sub eax,edi	        ; # of digits
	call type	        ;
	pop edi 	        ;
	pop edx 	        ;
next


; --------------------------------------------------------------
; 'parse'                                                 .forth
;
; Parses the input stream for a character or the end of the line
; --------------------------------------------------------------

code 'parse', parse	        ;
	mov edi,[tin]	        ; Pointer into TIB
	mov ecx,[tp]	        ;
	sub ecx,edi	        ;
	inc ecx 	        ;
	repe scasb	        ;
	dec edi 	        ;
	inc ecx 	        ;
	dup		        ;
	mov [esi],edi	        ; a
	repne scasb	        ;
	mov eax,edi	        ;
	dec eax 	        ;
	sub eax,[esi]	        ; n
	mov [tin],edi	        ;
next


; --------------------------------------------------------------
; 'reset'                                                 .forth
;
; Reset the stack to the default, empty position
; --------------------------------------------------------------

code 'reset', reset	        ; This word is used to reset the stack
	mov ecx, 2048		;
	mov esi, s0		;
	sub esi, 1024		;
.empty:	mov byte [esi], 0	;
	inc esi			;
	loop .empty		;
	mov esi, s0		;
	xor eax, eax		;
next



; --------------------------------------------------------------
; 'last'                                                  .forth
;
; Gives a pointer to the last entry in the dictionary
; --------------------------------------------------------------

code 'last', var_last
	upsh [last]
next


; --------------------------------------------------------------
; 'there'                                                 .forth
;
; Gives a pointer to the code buffer
; --------------------------------------------------------------

code 'there', there
	upsh blk
next


; --------------------------------------------------------------
; 'tib'                                                   .forth
;
; Gives the address of the TIB
; --------------------------------------------------------------

code 'tib', _tib
	upsh tib
next


; --------------------------------------------------------------
; 'word?'                                                 .forth
;
; This is a vectored word for handling errors during interpret
; and compile. 
; --------------------------------------------------------------

code 'word?', notfound
        upsh token0
        upsh [token0.size]
        call type
        call type
        upsh 10
        call emit
        mov dword [error], -1

	upsh [last]
	mov eax, [eax]
	add eax, 12
        stc
next


; --------------------------------------------------------------
; The default classes of words
;
; .data
;   : .data state @ if literal, then ;
; forth
;   : .forth state @ if compile ;; then execute ;
; macro
;   : .macro state @ if execute ;; then drop ;
; self
;   : .self execute ;
; --------------------------------------------------------------
code '.data', do_data
	cmp dword [state], -1
	jz literal
next

code '.forth', do_forth
	cmp dword [state], -1
	jz compile

code '.self', do_self
	upop edx
	jmp edx

code '.macro', do_macro
	cmp dword [state], -1
	jz do_self
	drop
ret
; --------------------------------------------------------------
var 'h0', h, h0		        ; h0     (pointer to HERE)
var 'base', base, 10	        ; base   current numeric base
var '>in', tin, 0	        ; >IN    Pointer into the TIB
var 'class', class, 0		;
var 'state', state, 0
var 'which', which, 0

error		dd 0		;
action		dd 0		; Handler for the current class
old_h		dd 0		;
d		dd d0		;
tail		dd 0		; Allow tail-calls?
source		dd 0	        ; Evaluate from RAM or KBD
tp		dd tib	        ; TP (pointer to input buffer)
bases		db 10,16,2,8,255; #dec $hex %bin &oct 'ascii
; --------------------------------------------------------------
include 'core.dict'
align 4
last  dd flast			; Last word in dictionary
flast dd forth_link	        ; Last word in 'forth'
; --------------------------------------------------------------


; --------------------------------------------------------------
; Error Messages
; --------------------------------------------------------------
boot0	db 'BOOT0:   End of bootstrap reached without a runtime action',10
	db '         defined. This is a fatal error; execution will now',10
	db '         suspend.',10
.size:	dd $-boot0

boot1	db 'BOOT1:   An error occured during bootstrap. The system may',10
	db '         not be stable or finish booting depending on the',10
	db '         word that resulted in the error.',10
.size:	dd $-boot1

token0	db 'TOKEN0:  Unable to find or convert the specified token to',10
	db '         a number. The token passed was: '
.size:	dd $-token0
; --------------------------------------------------------------
align 4
bootstrap:
	file "core.f"
.size dd $-bootstrap
; --------------------------------------------------------------
; section "bss"
; --------------------------------------------------------------
align 4
    rb 1024			; 256 cells  \ Total of 512 cells
s0  rb 1024			; 256 cells  / 2,048 bytes
tib rb 1024			; Text Input Buffer (1 KiB)
h0  rb 10000h			; Code ("heap")  (64k)
d0  rb 10000h			; Dictionary     (64k)
blk rb 10000h			; Block buffer   (64k)
; --------------------------------------------------------------
